"use client";

import React, { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "../components/ui/card";
import { Alert, AlertDescription, AlertTitle } from "../components/ui/alert";
import { Button } from "../components/ui/button";
import { AlertCircle, BellOff } from "lucide-react";
import { useDemoAuthContext } from "../context/demo-auth-context";
import { format } from 'date-fns';
import { useTranslation } from 'react-i18next';

interface ThresholdNotification {
  device_id: string;
  thresholdKey: string;
  message: string;
  timestamp: string;
}

interface NotificationsTabProps {
  isSystem: boolean;
}

const NotificationsTab: React.FC<NotificationsTabProps> = ({ isSystem }) => {
  const { user } = useDemoAuthContext();
  const { t, i18n } = useTranslation();
  const [notifications, setNotifications] = useState<ThresholdNotification[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Проверка перевода
  //console.log('NotificationsTab - Current language:', i18n.language);
  //console.log('NotificationsTab - Test translation:', t('notifications.message80', 'Device has reached 80% of max value'));

  useEffect(() => {
    const fetchNotifications = async () => {
      if (!user?.email) {
        setNotifications([]);
        setLoading(false);
        return;
      }
      setLoading(true);
      setError(null);
      try {
        const response = await fetch('/api/postgres/device-threshold-flags', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ email: user.email })
        });
        if (!response.ok) {
          const errorData = await response.json();
          throw new Error(errorData.error || t('errors.fetchFailed', 'Failed to fetch notifications'));
        }
        const data = await response.json();
        setNotifications(data.notifications || []);
      } catch (err: any) {
        setError(err.message || t('errors.fetchFailed', 'Failed to fetch notifications'));
        setNotifications([]);
      } finally {
        setLoading(false);
      }
    };
    fetchNotifications();
  }, [user, t]);

  // Функция для перевода ключей порогов
  const getThresholdTranslation = (thresholdKey: string) => {
    const thresholdMap: Record<string, string> = {
      '80%': t('notifications.threshold80', '80% threshold reached'),
      '85%': t('notifications.threshold85', '85% threshold reached'),
      '90%': t('notifications.threshold90', '90% threshold reached'),
      '100%': t('notifications.threshold100', '100% threshold reached'),
    };
    return thresholdMap[thresholdKey] || thresholdKey;
  };

  // Функция для перевода сообщений
  const getMessageTranslation = (message: string) => {
    //console.log('NotificationsTab - Original message:', message);
    
    // Динамический маппинг для любого устройства
    const deviceMatch = message.match(/Device (.+) reached (80%|85%|90%|100%) usage threshold\./);
    if (deviceMatch) {
      const [, deviceName, threshold] = deviceMatch;
      const thresholdKey = threshold as '80%' | '85%' | '90%' | '100%';
      const translationKey = `notifications.message${thresholdKey.replace('%', '')}`;
      //console.log('NotificationsTab - Translation key:', translationKey);
      const translated = t(translationKey, `Device ${deviceName} has reached ${threshold} of max value`);
      //console.log('NotificationsTab - Translated message:', translated);
      return translated;
    }
    
    //console.log('NotificationsTab - No match found, returning original');
    return message;
  };

  return (
    <Card className="w-full max-w-2xl mx-auto sm:max-w-full shadow-lg">
      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
        <CardTitle className="text-sm font-medium sm:text-xs">
          {isSystem ? t('notificationsTab.system', 'System Notifications') : t('notificationsTab.user', 'User Notifications')}
        </CardTitle>
      </CardHeader>
      <CardContent className="sm:px-2 sm:py-2">
        {loading ? (
          <p className="text-sm text-muted-foreground text-center py-4 sm:text-xs sm:py-2">{t('notificationsTab.loading', 'Loading notifications...')}</p>
        ) : error ? (
          <Alert variant="destructive">
            <AlertCircle className="h-4 w-4" />
            <AlertTitle>{t('common.error', 'Error')}</AlertTitle>
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        ) : notifications.length === 0 ? (
          <p className="text-sm text-muted-foreground text-center py-4 sm:text-xs sm:py-2">
            {t('notificationsTab.noNotifications', 'No notifications')}
          </p>
        ) : (
          <div className="space-y-4 sm:space-y-2 max-h-[60vh] overflow-y-auto">
            {notifications.map((notification, idx) => (
              <Alert key={`${notification.device_id}-${notification.thresholdKey}-${idx}`} className="sm:p-2">
                <AlertCircle className="h-4 w-4 sm:h-3 sm:w-3" />
                <AlertTitle className="sm:text-xs">
                  {notification.device_id} - {getThresholdTranslation(notification.thresholdKey)}
                </AlertTitle>
                <AlertDescription className="sm:text-xs">
                  {getMessageTranslation(notification.message)}
                  <div className="text-xs text-muted-foreground mt-1">
                    {notification.timestamp ? format(new Date(notification.timestamp), 'PPpp') : ''}
                  </div>
                </AlertDescription>
              </Alert>
            ))}
          </div>
        )}
      </CardContent>
    </Card>
  );
};

export default NotificationsTab;